/*
 *
 *  Copyright 2010-Present Peregrine Labs. All rights reserved.
 *  (c) 2010-Present Peregrine Labs a division of Peregrine Visual Storytelling Ltd.
 *  All rights reserved.
 *
 *  The coded instructions, statements, computer programs, and/or related
 *  material (collectively the "Data") in these files contain unpublished
 *  information proprietary to Peregrine Visual Storytelling Ltd. ("Peregrine")
 *  and/or its licensors, which is protected by U.S. and Canadian federal
 *  copyright law and by international treaties.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND. PEREGRINE
 *  DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED WARRANTIES
 *  INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF NON-INFRINGEMENT,
 *  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR ARISING FROM A COURSE
 *  OF DEALING, USAGE, OR TRADE PRACTICE. IN NO EVENT WILL PEREGRINE AND/OR ITS
 *  LICENSORS BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL,
 *  DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF PEREGRINE AND/OR ITS
 *  LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY OR PROBABILITY OF SUCH DAMAGES.
 *
 *
 */

#ifndef YETI_ATTRIBUTE_H
#define YETI_ATTRIBUTE_H

#include <vector>

#include "Common.h"
#include "Types.h"

YETI_NS_BEGIN

template <typename T>
class TypedAttribute;

using IntAttribute = TypedAttribute<IntType>;
using FloatAttribute = TypedAttribute<FloatType>;
using VectorAttribute = TypedAttribute<VectorType>;
using ColorAttribute = TypedAttribute<ColorType>;
using MatrixAttribute = TypedAttribute<MatrixType>;

class YETI_EXPORT Attribute
{
public:
    virtual size_t memUsage() const = 0;
    virtual int numElements() const = 0;
    virtual bool hidden() const = 0;
    virtual int flags() const = 0;

    virtual DataType dataType() const = 0;
    virtual DetailType detailType() const = 0;
    virtual TemporalType temporalType() const = 0;

    virtual std::shared_ptr<IntAttribute> asInt() const = 0;
    virtual std::shared_ptr<FloatAttribute> asFloat() const = 0;
    virtual std::shared_ptr<VectorAttribute> asVector() const = 0;
    virtual std::shared_ptr<ColorAttribute> asColor() const = 0;
    virtual std::shared_ptr<MatrixAttribute> asMatrix() const = 0;
};

template <typename T>
struct TypedAttributeData
{
    size_t numElements;
    const T *data;
};

template <typename T>
class YETI_EXPORT TypedAttribute : public Attribute
{
public:
    using DataType = TypedAttributeData<T>;
    using ValueType = T;
    using SampleTimes = std::vector<FloatType>;

    virtual DataType get(FloatType time) const = 0;
    virtual size_t typeSize() const = 0;
    virtual void getSampleTimes(SampleTimes &destination) const = 0;
};

using IntAttributeData = TypedAttribute<IntType>;
using FloatAttributeData = TypedAttribute<FloatType>;
using VectorAttributeData = TypedAttribute<VectorType>;
using ColorAttributeData = TypedAttribute<ColorType>;
using MatrixAttributeData = TypedAttribute<MatrixType>;

YETI_NS_END

#endif // YETI_ATTRIBUTE_H
